const axios = require('axios');
const crypto = require('crypto');

/**
 * PayChangu Payment Service
 * Handles payment initialization, verification, and webhook processing
 * Note: paychangu-js SDK is browser-only, so we use manual API integration on backend
 */

const PAYCHANGU_API_URL = process.env.PAYCHANGU_BASE_URL || 'https://api.paychangu.com';
const PUBLIC_KEY = process.env.PAYCHANGU_PUBLIC_KEY;
const SECRET_KEY = process.env.PAYCHANGU_SECRET_KEY;

/**
 * Generate transaction reference (compatible with paychangu-js format)
 */
const generateTxRef = () => {
  const timestamp = Date.now();
  const random = Math.random().toString(36).substring(2, 15);
  return `${timestamp}-${random}`;
};

/**
 * Initialize PayChangu payment
 * Returns payload for PayChangu popup SDK (used in frontend)
 */
const initializePayment = async (paymentData) => {
  if (!PUBLIC_KEY || !SECRET_KEY) {
    throw new Error('PayChangu credentials not configured');
  }

  try {
    console.log('[PAYCHANGU] Initializing payment:', {
      bookingId: paymentData.bookingId,
      amount: paymentData.amount,
      customerEmail: paymentData.customerEmail,
    });

    // Generate transaction reference
    const tx_ref = generateTxRef();
    
    // Build payload for PayChangu Popup SDK (browser-side)
    // Both callback_url and return_url are required by PayChangu SDK
    const payload = {
      public_key: PUBLIC_KEY,
      tx_ref: tx_ref,
      amount: paymentData.amount,
      currency: 'MWK',
      callback_url: paymentData.callbackUrl, // Required by SDK
      return_url: paymentData.returnUrl,     // For redirects after payment
      customer: {
        email: paymentData.customerEmail,
        first_name: paymentData.customerFirstName,
        last_name: paymentData.customerLastName,
      },
      customization: {
        title: 'Simbi Services',
        description: 'Car Wash Booking Payment',
      },
      meta: {
        booking_id: paymentData.bookingId,
        customer_phone: paymentData.customerPhone,
      },
    };

    console.log('[PAYCHANGU] Payment payload prepared for frontend PopupSDK');

    return {
      success: true,
      data: payload,
    };
  } catch (error) {
    console.error('[PAYCHANGU] Payment initialization error:', error.message);
    throw error;
  }
};

/**
 * Generate PayChangu signature
 * Signature = HMAC-SHA256(base64(secret_key + payload))
 */
const generateSignature = (payload) => {
  try {
    const message = JSON.stringify(payload);
    const signature = crypto
      .createHmac('sha256', SECRET_KEY)
      .update(message)
      .digest('hex');
    return signature;
  } catch (error) {
    console.error('[PAYCHANGU] Signature generation error:', error.message);
    throw error;
  }
};

/**
 * Verify PayChangu payment signature (webhook validation)
 */
const verifySignature = (payload, receivedSignature) => {
  try {
    const expectedSignature = generateSignature(payload);
    return crypto.timingSafeEqual(
      Buffer.from(expectedSignature),
      Buffer.from(receivedSignature)
    );
  } catch (error) {
    console.error('[PAYCHANGU] Signature verification error:', error.message);
    return false;
  }
};

/**
 * Handle PayChangu callback (webhook)
 * Processes payment status updates from PayChangu
 */
const handleCallback = async (callbackData, signature, db = null) => {
  try {
    console.log('[PAYCHANGU] Processing callback:', callbackData);

    // Verify signature (passed as separate parameter)
    if (!verifySignature(callbackData, signature)) {
      throw new Error('Invalid signature');
    }

    console.log('[PAYCHANGU] Signature verified successfully');

    // The tx_ref format is: "timestamp-randomstring"
    // We need to look up the bookingId from database since it's not in the tx_ref
    let bookingId = null;
    
    if (callbackData.meta && callbackData.meta.booking_id) {
      // If meta data contains booking_id, use it
      bookingId = callbackData.meta.booking_id;
      console.log('[PAYCHANGU] Retrieved booking_id from meta:', bookingId);
    } else if (db && callbackData.tx_ref) {
      // Fallback: Query database to find booking_id by tx_ref
      console.log('[PAYCHANGU] No booking_id in meta, querying database by tx_ref:', callbackData.tx_ref);
      const [payments] = await db.query(
        'SELECT booking_id FROM payments WHERE tx_ref = ?',
        [callbackData.tx_ref]
      );
      
      if (payments.length > 0) {
        bookingId = payments[0].booking_id;
        console.log('[PAYCHANGU] Retrieved booking_id from database:', bookingId);
      } else {
        console.error('[PAYCHANGU] No payment found for tx_ref:', callbackData.tx_ref);
        throw new Error('Payment record not found for tx_ref: ' + callbackData.tx_ref);
      }
    } else {
      console.error('[PAYCHANGU] Cannot determine booking_id - no meta and no database connection');
      throw new Error('Cannot determine booking_id from callback');
    }

    return {
      success: true,
      bookingId,
      tx_ref: callbackData.tx_ref || '',
      status: callbackData.status,
      transactionId: callbackData.transaction_id,
      amount: callbackData.amount,
      currency: callbackData.currency,
      timestamp: callbackData.timestamp,
    };
  } catch (error) {
    console.error('[PAYCHANGU] Callback processing error:', error.message);
    throw error;
  }
};

/**
 * Verify payment using PayChangu API
 * Makes direct API call to verify payment status
 */
const verifyPayment = async (txRef, maxRetries = 3) => {
  let lastError;
  
  for (let attempt = 1; attempt <= maxRetries; attempt++) {
    try {
      console.log(`[PAYCHANGU] Verifying payment (attempt ${attempt}/${maxRetries}):`, txRef);

      // Direct API call to PayChangu verification endpoint
      const response = await axios.get(
        `${PAYCHANGU_API_URL}/verify-payment`,
        {
          params: { tx_ref: txRef },
          headers: {
            'Authorization': `Bearer ${SECRET_KEY}`,
            'Content-Type': 'application/json',
          },
          timeout: 5000,
        }
      );

      console.log('[PAYCHANGU] Payment verification response:', response.data);

      if (response.data && response.data.status === 'success') {
        return {
          success: true,
          verified: true,
          transactionId: response.data.data?.id,
          amount: response.data.data?.amount,
          currency: response.data.data?.currency,
          status: response.data.data?.status,
        };
      }

      // If payment status is not final, could be processing
      if (attempt < maxRetries && response.data?.status === 'processing') {
        console.log(`[PAYCHANGU] Payment still processing, retrying...`);
        const delay = Math.pow(2, attempt - 1) * 1000; // Exponential backoff: 1s, 2s, 4s
        await new Promise(resolve => setTimeout(resolve, delay));
        continue;
      }

      return {
        success: false,
        verified: false,
        message: 'Payment verification failed',
      };
    } catch (error) {
      lastError = error;
      console.error(`[PAYCHANGU] Payment verification error (attempt ${attempt}):`, error.message);

      // Retry on network errors
      if (attempt < maxRetries) {
        const delay = Math.pow(2, attempt - 1) * 1000;
        await new Promise(resolve => setTimeout(resolve, delay));
        continue;
      }

      break;
    }
  }

  console.error('[PAYCHANGU] Payment verification failed after all retries:', lastError?.message);
  return {
    success: false,
    verified: false,
    message: 'Payment verification failed',
  };
};

module.exports = {
  initializePayment,
  generateSignature,
  verifySignature,
  handleCallback,
  verifyPayment,
};