const nodemailer = require('nodemailer');
const axios = require('axios');
const db = require('../config/database');

const getEmailTransporter = () => {
  return nodemailer.createTransport({
    host: process.env.EMAIL_HOST,
    port: parseInt(process.env.EMAIL_PORT || '587', 10),
    secure: false,
    auth: {
      user: process.env.EMAIL_USER,
      pass: process.env.EMAIL_PASS,
    },
  });
};

const sendEmail = async ({ to, subject, text }) => {
  if (!process.env.EMAIL_HOST || !process.env.EMAIL_USER || !process.env.EMAIL_PASS) {
    throw new Error('Email configuration is missing');
  }

  console.log(`[SEND EMAIL] Attempting to send email to: ${to}`);
  console.log(`[SEND EMAIL] Subject: ${subject}`);
  console.log(`[SEND EMAIL] Host: ${process.env.EMAIL_HOST}:${process.env.EMAIL_PORT}`);

  const transporter = getEmailTransporter();
  const from = process.env.EMAIL_FROM || process.env.EMAIL_USER;

  try {
    const info = await transporter.sendMail({
      from,
      to,
      subject,
      text,
    });
    console.log(`[SEND EMAIL] Success - Message ID: ${info.messageId}`);
    console.log(`[SEND EMAIL] Response:`, info.response);
    return info;
  } catch (error) {
    console.error(`[SEND EMAIL] Failed to send email`);
    console.error(`[SEND EMAIL] Error message:`, error.message);
    console.error(`[SEND EMAIL] Error code:`, error.code);
    throw error;
  }
};

const formatPhoneNumber = (phone) => {
  if (!phone) return phone;
  
  // Remove any non-digit characters
  const cleaned = phone.replace(/\D/g, '');
  console.log(`[FORMAT PHONE] Input: ${phone}, Cleaned: ${cleaned}, Length: ${cleaned.length}`);
  
  // If already has country code (11+ digits: 265 + 9 digits for Malawi)
  if (cleaned.length >= 11 && cleaned.startsWith('265')) {
    const formatted = `+${cleaned}`;
    console.log(`[FORMAT PHONE] Already has country code, returning: ${formatted}`);
    return formatted;
  }
  
  // If starts with 0, assume Malawi (+265) - remove leading 0 and add country code
  if (cleaned.startsWith('0') && cleaned.length === 10) {
    const formatted = `+265${cleaned.substring(1)}`;
    console.log(`[FORMAT PHONE] Local format detected, returning: ${formatted}`);
    return formatted;
  }
  
  // If 9 digits (no leading 0), assume Malawi (+265)
  if (cleaned.length === 9) {
    const formatted = `+265${cleaned}`;
    console.log(`[FORMAT PHONE] 9 digits detected, returning: ${formatted}`);
    return formatted;
  }
  
  // Otherwise, assume it's already properly formatted or has country code
  const formatted = `+${cleaned}`;
  console.log(`[FORMAT PHONE] Default case, returning: ${formatted}`);
  return formatted;
};

const sendSms = async ({ to, message }) => {
  if (!process.env.TEXTBEE_DEVICE_ID || !process.env.TEXTBEE_API_KEY) {
    throw new Error('TextBee configuration is missing');
  }

  const baseUrl = process.env.TEXTBEE_BASE_URL || 'https://api.textbee.dev/api/v1';
  const sendPath = process.env.TEXTBEE_SEND_PATH || `/gateway/devices/${process.env.TEXTBEE_DEVICE_ID}/send-sms`;
  const url = `${baseUrl}${sendPath}`;
  const headerName = process.env.TEXTBEE_API_HEADER || 'x-api-key';

  // Format phone number for TextBee API
  const formattedPhone = formatPhoneNumber(to);
  const fallbackPhone = formattedPhone.startsWith('+') ? formattedPhone.substring(1) : formattedPhone;
  
  console.log(`[SEND SMS] Original phone: ${to}`);
  console.log(`[SEND SMS] Formatted phone: ${formattedPhone}`);
  console.log(`[SEND SMS] URL: ${url}`);
  console.log(
    `[SEND SMS] Payload: { recipients: ["${formattedPhone}"], message: "${message}" }`
  );

  try {
    const response = await axios.post(
      url,
      {
        recipients: [formattedPhone],
        message,
      },
      {
        headers: {
          [headerName]: process.env.TEXTBEE_API_KEY,
          Accept: 'application/json',
        },
      }
    );
    console.log(`[SEND SMS] Success:`, response.data);
  } catch (error) {
    // Some providers/configurations may reject '+' prefix. Retry once without '+'.
    const isInvalidRecipients =
      error?.response?.status === 400 &&
      String(error?.response?.data?.error || '').toLowerCase().includes('invalid recipients');

    if (isInvalidRecipients && fallbackPhone && fallbackPhone !== formattedPhone) {
      console.warn('[SEND SMS] Invalid recipients with E.164, retrying without plus prefix');
      try {
        const retryResponse = await axios.post(
          url,
          {
            recipients: [fallbackPhone],
            message,
          },
          {
            headers: {
              [headerName]: process.env.TEXTBEE_API_KEY,
              Accept: 'application/json',
            },
          }
        );
        console.log(`[SEND SMS] Retry success:`, retryResponse.data);
        return;
      } catch (retryError) {
        console.error('[SEND SMS] Retry without plus prefix failed');
        error = retryError;
      }
    }

    console.error(`[SEND SMS] Failed to send SMS`);
    console.error(`[SEND SMS] Error message:`, error.message);
    if (error.response) {
      console.error(`[SEND SMS] Status:`, error.response.status);
      console.error(`[SEND SMS] Response data:`, error.response.data);
      console.error(`[SEND SMS] Response headers:`, error.response.headers);
    }
    
    // Log the full request details for debugging
    console.error(`[SEND SMS] Request details:`, {
      url: url,
      method: 'POST',
      headers: {
        [headerName]: process.env.TEXTBEE_API_KEY,
        Accept: 'application/json',
      },
      body: {
        recipients: [formattedPhone],
        message,
      }
    });
    
    throw error;
  }
};

const EXPO_PUSH_URL = 'https://exp.host/--/api/v2/push/send';
const EXPO_PUSH_RECEIPTS_URL = 'https://exp.host/--/api/v2/push/getReceipts';

const isExpoPushToken = (token) => {
  if (typeof token !== 'string') return false;
  const value = token.trim();
  if (!value) return false;
  // Support both legacy and current Expo token prefixes.
  return /^ExponentPushToken\[[^\]]+\]$/.test(value) || /^ExpoPushToken\[[^\]]+\]$/.test(value);
};

const getPreferenceColumnForType = (type = '') => {
  const normalized = String(type || '').toLowerCase();
  if (normalized.startsWith('booking_')) return 'booking_updates';
  if (normalized.startsWith('payment_')) return 'payment_updates';
  if (normalized.startsWith('payout_')) return 'payout_updates';
  if (normalized.startsWith('offer_') || normalized.startsWith('promo_')) return 'promo_updates';
  return 'system_updates';
};

const isPushAllowedForUser = async (userId, type) => {
  const [rows] = await db.query(
    `SELECT push_enabled, booking_updates, payment_updates, payout_updates, promo_updates, system_updates
     FROM notification_preferences
     WHERE user_id = ?
     LIMIT 1`,
    [userId]
  );

  if (rows.length === 0) return true;

  const pref = rows[0];
  if (!pref.push_enabled) return false;

  const prefColumn = getPreferenceColumnForType(type);
  return Boolean(pref[prefColumn]);
};

const getActivePushTokensForUser = async (userId) => {
  const [rows] = await db.query(
    `SELECT id, expo_push_token
     FROM user_push_tokens
     WHERE user_id = ? AND is_active = 1`,
    [userId]
  );
  return rows.filter((row) => isExpoPushToken(row.expo_push_token));
};

const deactivatePushTokensByValue = async (tokens = []) => {
  if (!Array.isArray(tokens) || tokens.length === 0) return;
  await db.query(
    `UPDATE user_push_tokens
     SET is_active = 0, updated_at = NOW()
     WHERE expo_push_token IN (?)`,
    [tokens]
  );
};

const sendExpoPushMessages = async (messages = []) => {
  if (!Array.isArray(messages) || messages.length === 0) {
    return { success: true, sent: 0, invalidTokens: [] };
  }

  const chunks = [];
  for (let i = 0; i < messages.length; i += 100) {
    chunks.push(messages.slice(i, i + 100));
  }

  const tickets = [];
  for (const chunk of chunks) {
    try {
      const response = await axios.post(EXPO_PUSH_URL, chunk, {
        timeout: 15000,
        headers: {
          Accept: 'application/json',
          'Accept-encoding': 'gzip, deflate',
          'Content-Type': 'application/json',
        },
      });
      const chunkData = Array.isArray(response?.data?.data) ? response.data.data : [];
      tickets.push(...chunkData);
    } catch (error) {
      console.error('[PUSH] Failed to send Expo push chunk:', error.message);
    }
  }

  const invalidTokens = [];
  const receiptIds = [];
  for (const ticket of tickets) {
    if (ticket?.status === 'error') {
      const detailsError = String(ticket?.details?.error || '');
      if (detailsError === 'DeviceNotRegistered') {
        if (ticket?.details?.expoPushToken) {
          invalidTokens.push(ticket.details.expoPushToken);
        }
      }
    }
    if (ticket?.id) receiptIds.push(ticket.id);
  }

  if (receiptIds.length > 0) {
    try {
      const receiptResponse = await axios.post(
        EXPO_PUSH_RECEIPTS_URL,
        { ids: receiptIds },
        {
          timeout: 15000,
          headers: {
            Accept: 'application/json',
            'Accept-encoding': 'gzip, deflate',
            'Content-Type': 'application/json',
          },
        }
      );
      const receiptData = receiptResponse?.data?.data || {};
      Object.values(receiptData).forEach((receipt) => {
        if (receipt?.status === 'error' && receipt?.details?.error === 'DeviceNotRegistered') {
          const badToken = receipt?.details?.expoPushToken;
          if (badToken) invalidTokens.push(badToken);
        }
      });
    } catch (error) {
      console.error('[PUSH] Failed to fetch Expo receipts:', error.message);
    }
  }

  const uniqueInvalidTokens = Array.from(new Set(invalidTokens));
  if (uniqueInvalidTokens.length > 0) {
    await deactivatePushTokensByValue(uniqueInvalidTokens);
  }

  return {
    success: true,
    sent: messages.length,
    invalidTokens: uniqueInvalidTokens,
  };
};

const sendPushToUser = async ({ userId, type, title, message, data = {} }) => {
  if (!userId || !title || !message) return { success: false, reason: 'missing_fields' };

  const pushAllowed = await isPushAllowedForUser(userId, type);
  if (!pushAllowed) return { success: true, skipped: true, reason: 'preferences_disabled' };

  const tokenRows = await getActivePushTokensForUser(userId);
  if (tokenRows.length === 0) return { success: true, skipped: true, reason: 'no_tokens' };

  const payload = tokenRows.map((row) => ({
    to: row.expo_push_token,
    sound: 'default',
    title,
    body: message,
    data: {
      type,
      ...data,
    },
    priority: 'high',
    channelId: 'default',
  }));

  return sendExpoPushMessages(payload);
};

const createNotificationAndPush = async ({ userId, type, title, message, data = {} }) => {
  if (!userId || !type || !title || !message) return;

  await db.query(
    `INSERT INTO notifications (user_id, type, title, message, data)
     VALUES (?, ?, ?, ?, ?)`,
    [userId, type, title, message, JSON.stringify(data || {})]
  );

  await sendPushToUser({ userId, type, title, message, data });
};

module.exports = {
  sendEmail,
  sendSms,
  isExpoPushToken,
  sendPushToUser,
  createNotificationAndPush,
};
