const rateLimit = require('express-rate-limit');

/**
 * Rate limiter for webhook endpoints
 * Allows 100 requests per minute per IP
 */
const webhookLimiter = rateLimit({
  windowMs: 60 * 1000, // 1 minute
  max: 100, // 100 requests per minute
  message: 'Too many webhook requests from this IP, please try again later',
  standardHeaders: true, // Return rate limit info in `RateLimit-*` headers
  legacyHeaders: false, // Disable `X-RateLimit-*` headers
  skip: (req) => {
    // Skip rate limiting for non-POST requests
    return req.method !== 'POST';
  },
});

/**
 * Rate limiter for API endpoints
 * Development mode: Very high limits for testing
 * Production: Will need stricter limits
 */
const apiLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: process.env.NODE_ENV === 'production' ? 1000 : 10000, // 10000 for dev, 1000 for prod
  message: 'Too many requests from this IP, please try again later',
  standardHeaders: true,
  legacyHeaders: false,
  skip: (req) => {
    // Skip rate limiting in development mode if needed
    return process.env.NODE_ENV === 'development';
  },
});

module.exports = {
  webhookLimiter,
  apiLimiter,
};
