const db = require('../config/database');
const { asyncHandler } = require('../middleware/errorHandler');

/**
 * @desc    Get all services with filters
 * @route   GET /api/v1/services
 * @access  Public
 */
exports.getAllServices = asyncHandler(async (req, res) => {
  const { 
    category, 
    city, 
    min_price, 
    max_price, 
    detailer_id, 
    search,
    latitude,
    longitude,
    radius = 50, // km
    page = 1, 
    limit = 20 
  } = req.query;

  let query = `
    SELECT 
      s.id, s.name, s.description, s.price, s.duration_minutes, s.is_active, s.location_type,
      sc.name AS category_name,
      d.id AS detailer_id, d.business_name, d.city, d.address, 
      d.latitude, d.longitude, d.rating_average, d.total_reviews, d.is_verified
  `;

  // Add distance calculation if location provided
  if (latitude && longitude) {
    query += `, (6371 * acos(cos(radians(?)) * cos(radians(d.latitude)) * 
               cos(radians(d.longitude) - radians(?)) + sin(radians(?)) * 
               sin(radians(d.latitude)))) AS distance `;
  }

  query += `
    FROM services s
    INNER JOIN detailers d ON s.detailer_id = d.id
    INNER JOIN service_categories sc ON s.category_id = sc.id
    WHERE s.deleted_at IS NULL 
      AND s.is_active = TRUE 
      AND d.deleted_at IS NULL
      AND d.is_available = TRUE
  `;

  const params = [];

  // Add location params if provided
  if (latitude && longitude) {
    params.push(parseFloat(latitude), parseFloat(longitude), parseFloat(latitude));
  }

  // Apply filters
  if (category) {
    query += ' AND sc.id = ?';
    params.push(category);
  }

  if (city) {
    query += ' AND d.city LIKE ?';
    params.push(`%${city}%`);
  }

  if (min_price) {
    query += ' AND s.price >= ?';
    params.push(parseFloat(min_price));
  }

  if (max_price) {
    query += ' AND s.price <= ?';
    params.push(parseFloat(max_price));
  }

  if (detailer_id) {
    query += ' AND d.id = ?';
    params.push(detailer_id);
  }

  if (search) {
    query += ' AND (s.name LIKE ? OR s.description LIKE ? OR d.business_name LIKE ?)';
    params.push(`%${search}%`, `%${search}%`, `%${search}%`);
  }

  // Distance filter
  if (latitude && longitude && radius) {
    query += ' HAVING distance <= ?';
    params.push(parseFloat(radius));
  }

  // Order by
  if (latitude && longitude) {
    query += ' ORDER BY distance ASC';
  } else {
    query += ' ORDER BY d.rating_average DESC, s.price ASC';
  }

  // Pagination
  const offset = (parseInt(page) - 1) * parseInt(limit);
  query += ' LIMIT ? OFFSET ?';
  params.push(parseInt(limit), offset);

  const [services] = await db.query(query, params);

  // Get total count
  let countQuery = `
    SELECT COUNT(DISTINCT s.id) as total
    FROM services s
    INNER JOIN detailers d ON s.detailer_id = d.id
    INNER JOIN service_categories sc ON s.category_id = sc.id
    WHERE s.deleted_at IS NULL 
      AND s.is_active = TRUE 
      AND d.deleted_at IS NULL
      AND d.is_available = TRUE
  `;

  const countParams = [];

  if (category) {
    countQuery += ' AND sc.id = ?';
    countParams.push(category);
  }

  if (city) {
    countQuery += ' AND d.city LIKE ?';
    countParams.push(`%${city}%`);
  }

  if (min_price) {
    countQuery += ' AND s.price >= ?';
    countParams.push(parseFloat(min_price));
  }

  if (max_price) {
    countQuery += ' AND s.price <= ?';
    countParams.push(parseFloat(max_price));
  }

  if (detailer_id) {
    countQuery += ' AND d.id = ?';
    countParams.push(detailer_id);
  }

  if (search) {
    countQuery += ' AND (s.name LIKE ? OR s.description LIKE ? OR d.business_name LIKE ?)';
    countParams.push(`%${search}%`, `%${search}%`, `%${search}%`);
  }

  const [countResult] = await db.query(countQuery, countParams);
  const total = countResult[0].total;

  res.json({
    success: true,
    data: services,
    pagination: {
      page: parseInt(page),
      limit: parseInt(limit),
      total,
      pages: Math.ceil(total / parseInt(limit))
    }
  });
});

/**
 * @desc    Get service categories
 * @route   GET /api/v1/services/categories
 * @access  Public
 */
exports.getCategories = asyncHandler(async (req, res) => {
  const [categories] = await db.query(
    'SELECT id, name, description, icon FROM service_categories WHERE is_active = TRUE ORDER BY name'
  );

  res.json({
    success: true,
    data: categories
  });
});

/**
 * @desc    Get service by ID
 * @route   GET /api/v1/services/:id
 * @access  Public
 */
exports.getServiceById = asyncHandler(async (req, res) => {
  const { id } = req.params;

  const [services] = await db.query(
    `SELECT 
      s.*, 
      sc.name AS category_name,
      d.id AS detailer_id, d.business_name, d.owner_name, d.phone, 
      d.address, d.city, d.state, d.latitude, d.longitude, 
      d.description AS detailer_description, d.profile_image,
      d.rating_average, d.total_reviews, d.is_verified, d.is_available
    FROM services s
    INNER JOIN detailers d ON s.detailer_id = d.id
    INNER JOIN service_categories sc ON s.category_id = sc.id
    WHERE s.id = ? AND s.deleted_at IS NULL`,
    [id]
  );

  if (services.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Service not found'
    });
  }

  res.json({
    success: true,
    data: services[0]
  });
});

/**
 * @desc    Create new service
 * @route   POST /api/v1/services
 * @access  Private (Detailer only)
 */
exports.createService = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { category_id, name, description, price, duration_minutes, location_type } = req.body;

  // Get detailer ID
  const [detailers] = await db.query(
    'SELECT id FROM detailers WHERE user_id = ? AND deleted_at IS NULL',
    [userId]
  );

  if (detailers.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Detailer profile not found'
    });
  }

  const detailerId = detailers[0].id;

  // Insert service
  const [result] = await db.query(
    `INSERT INTO services (detailer_id, category_id, name, description, price, duration_minutes, location_type) 
     VALUES (?, ?, ?, ?, ?, ?, ?)`,
    [detailerId, category_id, name, description, price, duration_minutes, location_type || 'any']
  );

  res.status(201).json({
    success: true,
    message: 'Service created successfully',
    data: { id: result.insertId }
  });
});

/**
 * @desc    Update service
 * @route   PUT /api/v1/services/:id
 * @access  Private (Detailer only)
 */
exports.updateService = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const userId = req.user.id;
  const { category_id, name, description, price, duration_minutes, is_active } = req.body;

  // Verify ownership
  const [services] = await db.query(
    `SELECT s.id FROM services s
     INNER JOIN detailers d ON s.detailer_id = d.id
     WHERE s.id = ? AND d.user_id = ? AND s.deleted_at IS NULL`,
    [id, userId]
  );

  if (services.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Service not found or you do not have permission to update it'
    });
  }

  // Update service
  await db.query(
    `UPDATE services 
     SET category_id = ?, name = ?, description = ?, price = ?, duration_minutes = ?, is_active = ?
     WHERE id = ?`,
    [category_id, name, description, price, duration_minutes, is_active, id]
  );

  res.json({
    success: true,
    message: 'Service updated successfully'
  });
});

/**
 * @desc    Delete service (soft delete)
 * @route   DELETE /api/v1/services/:id
 * @access  Private (Detailer only)
 */
exports.deleteService = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const userId = req.user.id;

  // Verify ownership
  const [services] = await db.query(
    `SELECT s.id FROM services s
     INNER JOIN detailers d ON s.detailer_id = d.id
     WHERE s.id = ? AND d.user_id = ? AND s.deleted_at IS NULL`,
    [id, userId]
  );

  if (services.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Service not found or you do not have permission to delete it'
    });
  }

  // Soft delete
  await db.query(
    'UPDATE services SET deleted_at = NOW() WHERE id = ?',
    [id]
  );

  res.json({
    success: true,
    message: 'Service deleted successfully'
  });
});

/**
 * @desc    Get detailer's own services
 * @route   GET /api/v1/services/my-services
 * @access  Private (Detailer only)
 */
exports.getMyServices = asyncHandler(async (req, res) => {
  const userId = req.user.id;

  // Get detailer ID
  const [detailers] = await db.query(
    'SELECT id FROM detailers WHERE user_id = ?',
    [userId]
  );

  if (detailers.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Detailer profile not found'
    });
  }

  const detailerId = detailers[0].id;

  // Get all services for this detailer
  const [services] = await db.query(
    `SELECT 
      s.id, s.name, s.description, s.price, s.duration_minutes, 
      s.category_id, s.is_active, s.created_at, s.updated_at,
      COUNT(DISTINCT b.id) as total_bookings
    FROM services s
    LEFT JOIN bookings b ON s.id = b.service_id AND b.deleted_at IS NULL
    WHERE s.detailer_id = ? AND s.deleted_at IS NULL
    GROUP BY s.id
    ORDER BY s.created_at DESC`,
    [detailerId]
  );

  res.json({
    success: true,
    data: services
  });
});
