const db = require('../config/database');
const { asyncHandler } = require('../middleware/errorHandler');
const { sendPushToUser } = require('../services/notificationService');

/**
 * @desc    Create review for completed booking
 * @route   POST /api/v1/reviews
 * @access  Private (Client only)
 */
exports.createReview = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { booking_id, rating, review_text } = req.body;

  if (!rating || rating < 1 || rating > 5) {
    return res.status(400).json({
      success: false,
      message: 'Rating must be between 1 and 5'
    });
  }

  const connection = await db.getConnection();

  try {
    await connection.beginTransaction();

    // Get client ID
    const [clients] = await connection.query(
      'SELECT id FROM clients WHERE user_id = ?',
      [userId]
    );

    if (clients.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Client profile not found'
      });
    }

    const clientId = clients[0].id;

    // Verify booking
    const [bookings] = await connection.query(
      `SELECT id, detailer_id, status 
       FROM bookings 
       WHERE id = ? AND client_id = ? AND deleted_at IS NULL`,
      [booking_id, clientId]
    );

    if (bookings.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Booking not found or access denied'
      });
    }

    const booking = bookings[0];

    if (booking.status !== 'completed') {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'You can only review completed bookings'
      });
    }

    // Check if review already exists
    const [existingReviews] = await connection.query(
      'SELECT id FROM reviews WHERE booking_id = ?',
      [booking_id]
    );

    if (existingReviews.length > 0) {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'You have already reviewed this booking'
      });
    }

    // Create review
    const [result] = await connection.query(
      `INSERT INTO reviews (booking_id, client_id, detailer_id, rating, review_text)
       VALUES (?, ?, ?, ?, ?)`,
      [booking_id, clientId, booking.detailer_id, rating, review_text]
    );

    // Update detailer rating
    const [ratingStats] = await connection.query(
      `SELECT 
        COUNT(*) as total_reviews,
        AVG(rating) as avg_rating
       FROM reviews
       WHERE detailer_id = ? AND is_visible = TRUE`,
      [booking.detailer_id]
    );

    await connection.query(
      `UPDATE detailers 
       SET rating_average = ?, total_reviews = ?
       WHERE id = ?`,
      [ratingStats[0].avg_rating, ratingStats[0].total_reviews, booking.detailer_id]
    );

    // Notify detailer
    const [detailers] = await connection.query(
      'SELECT user_id FROM detailers WHERE id = ?',
      [booking.detailer_id]
    );

    if (detailers.length > 0) {
      await connection.query(
        `INSERT INTO notifications (user_id, type, title, message, data)
         VALUES (?, 'new_review', 'New Review Received', ?, ?)`,
        [
          detailers[0].user_id,
          `You received a ${rating}-star review`,
          JSON.stringify({ review_id: result.insertId, rating })
        ]
      );

      try {
        await sendPushToUser({
          userId: detailers[0].user_id,
          type: 'new_review',
          title: 'New Review Received',
          message: `You received a ${rating}-star review`,
          data: { review_id: result.insertId, rating },
        });
      } catch (pushError) {
        console.error('[PUSH] Failed to send new_review push:', pushError.message);
      }
    }

    await connection.commit();

    res.status(201).json({
      success: true,
      message: 'Review submitted successfully',
      data: { id: result.insertId }
    });

  } catch (error) {
    await connection.rollback();
    throw error;
  } finally {
    connection.release();
  }
});

/**
 * @desc    Update review
 * @route   PUT /api/v1/reviews/:id
 * @access  Private (Client only)
 */
exports.updateReview = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const userId = req.user.id;
  const { rating, review_text } = req.body;

  if (rating && (rating < 1 || rating > 5)) {
    return res.status(400).json({
      success: false,
      message: 'Rating must be between 1 and 5'
    });
  }

  const connection = await db.getConnection();

  try {
    await connection.beginTransaction();

    // Verify ownership
    const [reviews] = await connection.query(
      `SELECT r.id, r.detailer_id 
       FROM reviews r
       INNER JOIN clients c ON r.client_id = c.id
       WHERE r.id = ? AND c.user_id = ?`,
      [id, userId]
    );

    if (reviews.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Review not found or access denied'
      });
    }

    const review = reviews[0];

    // Update review
    await connection.query(
      'UPDATE reviews SET rating = ?, review_text = ? WHERE id = ?',
      [rating, review_text, id]
    );

    // Recalculate detailer rating
    const [ratingStats] = await connection.query(
      `SELECT 
        COUNT(*) as total_reviews,
        AVG(rating) as avg_rating
       FROM reviews
       WHERE detailer_id = ? AND is_visible = TRUE`,
      [review.detailer_id]
    );

    await connection.query(
      `UPDATE detailers 
       SET rating_average = ?, total_reviews = ?
       WHERE id = ?`,
      [ratingStats[0].avg_rating, ratingStats[0].total_reviews, review.detailer_id]
    );

    await connection.commit();

    res.json({
      success: true,
      message: 'Review updated successfully'
    });

  } catch (error) {
    await connection.rollback();
    throw error;
  } finally {
    connection.release();
  }
});

/**
 * @desc    Get detailer reviews
 * @route   GET /api/v1/reviews/detailer/:detailerId
 * @access  Public
 */
exports.getDetailerReviews = asyncHandler(async (req, res) => {
  const { detailerId } = req.params;
  const { page = 1, limit = 10 } = req.query;

  const [reviews] = await db.query(
    `SELECT 
      r.id, r.rating, r.review_text, r.created_at,
      c.first_name, c.last_name, c.profile_image
    FROM reviews r
    INNER JOIN clients c ON r.client_id = c.id
    WHERE r.detailer_id = ? AND r.is_visible = TRUE
    ORDER BY r.created_at DESC
    LIMIT ? OFFSET ?`,
    [detailerId, parseInt(limit), (parseInt(page) - 1) * parseInt(limit)]
  );

  // Get review statistics
  const [stats] = await db.query(
    `SELECT 
      COUNT(*) as total_reviews,
      AVG(rating) as average_rating,
      SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as five_star,
      SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as four_star,
      SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as three_star,
      SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as two_star,
      SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as one_star
    FROM reviews
    WHERE detailer_id = ? AND is_visible = TRUE`,
    [detailerId]
  );

  res.json({
    success: true,
    data: {
      reviews,
      statistics: stats[0]
    }
  });
});

/**
 * @desc    Get detailer's own reviews
 * @route   GET /api/v1/reviews/my-reviews
 * @access  Private (Detailer only)
 */
exports.getMyReviews = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { page = 1, limit = 10 } = req.query;

  // Get detailer ID
  const [detailers] = await db.query(
    'SELECT id FROM detailers WHERE user_id = ?',
    [userId]
  );

  if (detailers.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Detailer profile not found'
    });
  }

  const detailerId = detailers[0].id;

  // Get reviews for this detailer
  const [reviews] = await db.query(
    `SELECT 
      r.id, r.rating, r.review_text, r.created_at, r.is_visible,
      c.first_name, c.last_name, c.profile_image,
      b.id as booking_id,
      s.name as service_name
    FROM reviews r
    INNER JOIN clients c ON r.client_id = c.id
    INNER JOIN bookings b ON r.booking_id = b.id
    INNER JOIN services s ON b.service_id = s.id
    WHERE r.detailer_id = ?
    ORDER BY r.created_at DESC
    LIMIT ? OFFSET ?`,
    [detailerId, parseInt(limit), (parseInt(page) - 1) * parseInt(limit)]
  );

  // Get review statistics
  const [stats] = await db.query(
    `SELECT 
      COUNT(*) as total_reviews,
      AVG(rating) as average_rating,
      SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as five_star,
      SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as four_star,
      SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as three_star,
      SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as two_star,
      SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as one_star
    FROM reviews
    WHERE detailer_id = ?`,
    [detailerId]
  );

  res.json({
    success: true,
    data: {
      reviews,
      statistics: stats[0]
    }
  });
});
