const db = require('../config/database');
const { asyncHandler } = require('../middleware/errorHandler');
const upload = require('../config/upload');

const normalizePhone = (value) => {
  const cleaned = String(value || '').replace(/\D/g, '');
  if (!cleaned) return '';
  if (cleaned.startsWith('265') && cleaned.length >= 11) return `+${cleaned}`;
  if (cleaned.startsWith('0') && cleaned.length === 10) return `+265${cleaned.substring(1)}`;
  if (cleaned.length === 9) return `+265${cleaned}`;
  return `+${cleaned}`;
};

const getDetailerIdByUser = async (userId) => {
  const [detailers] = await db.query('SELECT id FROM detailers WHERE user_id = ?', [userId]);
  return detailers.length > 0 ? detailers[0].id : null;
};

const getPayoutBalance = async (detailerId) => {
  const [creditedRows] = await db.query(
    `SELECT COALESCE(SUM(b.amount - COALESCE(b.commission_amount, 0)), 0) AS credited
     FROM bookings b
     LEFT JOIN payments p ON p.booking_id = b.id
     WHERE b.detailer_id = ?
       AND b.status = 'completed'
       AND b.deleted_at IS NULL
       AND p.status = 'verified'`,
    [detailerId]
  );

  const [requestedRows] = await db.query(
    `SELECT COALESCE(SUM(amount), 0) AS requested
     FROM payout_requests
     WHERE detailer_id = ?
       AND status IN ('pending', 'processing', 'approved', 'paid')`,
    [detailerId]
  );

  const credited = Number(creditedRows?.[0]?.credited || 0);
  const requested = Number(requestedRows?.[0]?.requested || 0);
  const available = Math.max(0, credited - requested);
  return { credited, requested, available };
};

const runAutoPayoutIfDue = async (detailerId) => {
  const [rows] = await db.query(
    `SELECT id, auto_payout_enabled, auto_payout_frequency, auto_payout_day_of_week,
            auto_payout_day_of_month, auto_payout_min_amount, last_auto_payout_at
     FROM detailer_payout_settings
     WHERE detailer_id = ?
     LIMIT 1`,
    [detailerId]
  );

  if (rows.length === 0) return;
  const settings = rows[0];
  if (!settings.auto_payout_enabled) return;

  const now = new Date();
  const lastRun = settings.last_auto_payout_at ? new Date(settings.last_auto_payout_at) : null;
  let shouldRun = false;

  if (settings.auto_payout_frequency === 'daily') {
    shouldRun = !lastRun || lastRun.toDateString() !== now.toDateString();
  } else if (settings.auto_payout_frequency === 'weekly') {
    const dow = Number(settings.auto_payout_day_of_week ?? 1);
    const isCorrectDay = now.getDay() === dow;
    shouldRun = isCorrectDay && (!lastRun || (now - lastRun) >= 6 * 24 * 60 * 60 * 1000);
  } else if (settings.auto_payout_frequency === 'monthly') {
    const dom = Number(settings.auto_payout_day_of_month ?? 1);
    const isCorrectDay = now.getDate() === dom;
    shouldRun =
      isCorrectDay &&
      (!lastRun ||
        lastRun.getMonth() !== now.getMonth() ||
        lastRun.getFullYear() !== now.getFullYear());
  }

  if (!shouldRun) return;

  const { available } = await getPayoutBalance(detailerId);
  const minAmount = Number(settings.auto_payout_min_amount || 0);
  if (!Number.isFinite(available) || available <= 0 || available < minAmount) {
    await db.query(
      'UPDATE detailer_payout_settings SET last_auto_payout_at = NOW() WHERE id = ?',
      [settings.id]
    );
    return;
  }

  await db.query(
    `INSERT INTO payout_requests (detailer_id, amount, request_type, status, notes, requested_at)
     VALUES (?, ?, 'auto', 'pending', ?, NOW())`,
    [detailerId, available, `Auto payout (${settings.auto_payout_frequency})`]
  );
  await db.query(
    'UPDATE detailer_payout_settings SET last_auto_payout_at = NOW() WHERE id = ?',
    [settings.id]
  );
};

/**
 * @desc    Get all detailers with filters
 * @route   GET /api/v1/detailers
 * @access  Public
 */
exports.getAllDetailers = asyncHandler(async (req, res) => {
  const { 
    city, 
    min_rating, 
    verified_only,
    latitude,
    longitude,
    radius = 50,
    page = 1, 
    limit = 20 
  } = req.query;

  let query = `
    SELECT 
      d.id, d.business_name, d.owner_name, d.phone, d.address, d.city, d.state,
      d.latitude, d.longitude, d.description,
      dp_primary.photo_url AS primary_photo_url,
      d.is_verified, d.is_available, d.rating_average, d.total_reviews,
      d.total_bookings, d.completed_bookings
  `;

  // Add distance calculation if location provided
  if (latitude && longitude) {
    query += `, (6371 * acos(cos(radians(?)) * cos(radians(d.latitude)) * 
               cos(radians(d.longitude) - radians(?)) + sin(radians(?)) * 
               sin(radians(d.latitude)))) AS distance `;
  }

  query += `
    FROM detailers d
    LEFT JOIN detailer_photos dp_primary
      ON dp_primary.detailer_id = d.id
      AND dp_primary.photo_type = 'gallery'
      AND dp_primary.is_deleted = 0
      AND dp_primary.is_primary = 1
    WHERE d.deleted_at IS NULL AND d.is_available = TRUE
  `;

  const params = [];

  // Add location params if provided
  if (latitude && longitude) {
    params.push(parseFloat(latitude), parseFloat(longitude), parseFloat(latitude));
  }

  // Apply filters
  if (city) {
    query += ' AND d.city LIKE ?';
    params.push(`%${city}%`);
  }

  if (min_rating) {
    query += ' AND d.rating_average >= ?';
    params.push(parseFloat(min_rating));
  }

  if (verified_only === 'true') {
    query += ' AND d.is_verified = TRUE';
  }

  // Distance filter
  if (latitude && longitude && radius) {
    query += ' HAVING distance <= ?';
    params.push(parseFloat(radius));
  }

  // Order by
  if (latitude && longitude) {
    query += ' ORDER BY distance ASC';
  } else {
    query += ' ORDER BY d.rating_average DESC, d.total_reviews DESC';
  }

  // Pagination
  const offset = (parseInt(page) - 1) * parseInt(limit);
  query += ' LIMIT ? OFFSET ?';
  params.push(parseInt(limit), offset);

  const [detailers] = await db.query(query, params);

  res.json({
    success: true,
    data: detailers
  });
});

/**
 * @desc    Get detailer by ID
 * @route   GET /api/v1/detailers/:id
 * @access  Public
 */
exports.getDetailerById = asyncHandler(async (req, res) => {
  const { id } = req.params;

  const [detailers] = await db.query(
    `SELECT 
      d.*,
      u.email as user_email,
      COUNT(DISTINCT s.id) as total_services,
      COUNT(DISTINCT dp.id) as total_photos
    FROM detailers d
    INNER JOIN users u ON d.user_id = u.id
    LEFT JOIN services s ON d.id = s.detailer_id AND s.is_active = TRUE AND s.deleted_at IS NULL
    LEFT JOIN detailer_photos dp ON d.id = dp.detailer_id AND dp.photo_type = 'gallery' AND dp.is_deleted = 0
    WHERE d.id = ? AND d.deleted_at IS NULL AND u.deleted_at IS NULL
    GROUP BY d.id`,
    [id]
  );

  if (detailers.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Detailer not found'
    });
  }

  res.json({
    success: true,
    data: detailers[0]
  });
});

/**
 * @desc    Update detailer profile
 * @route   PUT /api/v1/detailers/profile
 * @access  Private (Detailer only)
 */
exports.updateProfile = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const {
    business_name,
    owner_name,
    phone,
    address,
    city,
    state,
    latitude,
    longitude,
    description
  } = req.body;

  // Get detailer ID
  const [detailers] = await db.query(
    'SELECT id FROM detailers WHERE user_id = ?',
    [userId]
  );

  if (detailers.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Detailer profile not found'
    });
  }

  const detailerId = detailers[0].id;

  // Prepare update data
  const updateData = {
    business_name,
    owner_name,
    phone,
    address,
    city,
    state,
    latitude,
    longitude,
    description
  };

  // Add profile image if uploaded
  if (req.file) {
    updateData.profile_image = `/uploads/profile-images/${req.file.filename}`;
  }

  // Build update query
  const fields = Object.keys(updateData).filter(key => updateData[key] !== undefined);
  const values = fields.map(key => updateData[key]);
  values.push(detailerId);

  const setClause = fields.map(key => `${key} = ?`).join(', ');

  await db.query(
    `UPDATE detailers SET ${setClause} WHERE id = ?`,
    values
  );

  // Get updated profile
  const [updatedProfile] = await db.query(
    `SELECT 
      d.*, 
      u.email, u.role
    FROM detailers d
    INNER JOIN users u ON d.user_id = u.id
    WHERE d.id = ?`,
    [detailerId]
  );

  res.json({
    success: true,
    message: 'Profile updated successfully',
    data: updatedProfile[0]
  });
});

/**
 * @desc    Update detailer availability
 * @route   PUT /api/v1/detailers/availability
 * @access  Private (Detailer only)
 */
exports.updateAvailability = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { is_available } = req.body;

  await db.query(
    `UPDATE detailers SET is_available = ? 
     WHERE user_id = ? AND deleted_at IS NULL`,
    [is_available, userId]
  );

  res.json({
    success: true,
    message: `Availability updated to ${is_available ? 'available' : 'unavailable'}`
  });
});

/**
 * @desc    Get detailer dashboard statistics
 * @route   GET /api/v1/detailers/dashboard/stats
 * @access  Private (Detailer only)
 */
exports.getDashboardStats = asyncHandler(async (req, res) => {
  const userId = req.user.id;

  // Get detailer ID
  const [detailers] = await db.query(
    'SELECT id FROM detailers WHERE user_id = ?',
    [userId]
  );

  if (detailers.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Detailer profile not found'
    });
  }

  const detailerId = detailers[0].id;
  await runAutoPayoutIfDue(detailerId);

  // Get booking statistics
  const [bookingStats] = await db.query(
    `SELECT 
      COUNT(*) as total_bookings,
      SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_bookings,
      SUM(CASE WHEN status = 'accepted' THEN 1 ELSE 0 END) as accepted_bookings,
      SUM(CASE WHEN status = 'assigned' THEN 1 ELSE 0 END) as assigned_bookings,
      SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_bookings,
      SUM(CASE WHEN status = 'completed' THEN amount - commission_amount ELSE 0 END) as total_earnings,
      SUM(CASE WHEN status = 'completed' AND MONTH(completed_at) = MONTH(CURRENT_DATE) AND YEAR(completed_at) = YEAR(CURRENT_DATE) THEN amount - commission_amount ELSE 0 END) as monthly_earnings
    FROM bookings
    WHERE detailer_id = ? AND deleted_at IS NULL`,
    [detailerId]
  );

  // Get average rating
  const [ratingData] = await db.query(
    `SELECT AVG(rating) as average_rating
    FROM reviews
    WHERE detailer_id = ?`,
    [detailerId]
  );

  // Get recent bookings
  const [recentBookings] = await db.query(
    `SELECT 
      b.id, b.booking_number, b.status, b.booking_date, b.booking_time, b.amount,
      CONCAT(c.first_name, ' ', c.last_name) as client_name,
      s.name as service_name
    FROM bookings b
    INNER JOIN clients c ON b.client_id = c.id
    INNER JOIN services s ON b.service_id = s.id
    WHERE b.detailer_id = ? AND b.deleted_at IS NULL
    ORDER BY b.created_at DESC
    LIMIT 5`,
    [detailerId]
  );

  const payoutBalance = await getPayoutBalance(detailerId);
  const [payoutRows] = await db.query(
    `SELECT
      COALESCE(SUM(CASE WHEN status = 'paid' THEN amount ELSE 0 END), 0) AS total_paid_out,
      COALESCE(SUM(CASE WHEN status IN ('pending', 'processing', 'approved') THEN amount ELSE 0 END), 0) AS pending_payouts
    FROM payout_requests
    WHERE detailer_id = ?`,
    [detailerId]
  );

  res.json({
    success: true,
    data: {
      ...bookingStats[0],
      available_balance: payoutBalance.available,
      total_requested: payoutBalance.requested,
      total_paid_out: Number(payoutRows?.[0]?.total_paid_out || 0),
      pending_payouts: Number(payoutRows?.[0]?.pending_payouts || 0),
      average_rating: ratingData[0].average_rating,
      recent_bookings: recentBookings
    }
  });
});

/**
 * @desc    Upload detailer photos (gallery)
 * @route   POST /api/v1/detailers/photos
 * @access  Private (Detailer only)
 */
exports.uploadPhotos = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const photos = req.files;

  console.log(`[UPLOAD PHOTOS] User ${userId}, Files count: ${photos?.length || 0}`);

  if (!photos || photos.length === 0) {
    console.log('[UPLOAD PHOTOS] No photos in request');
    return res.status(400).json({
      success: false,
      message: 'No photos uploaded'
    });
  }

  if (photos.length > 4) {
    console.log(`[UPLOAD PHOTOS] Too many photos: ${photos.length}`);
    return res.status(400).json({
      success: false,
      message: 'Maximum 4 photos allowed'
    });
  }

  // Get detailer ID
  const [detailers] = await db.query(
    'SELECT id FROM detailers WHERE user_id = ?',
    [userId]
  );

  if (detailers.length === 0) {
    console.log(`[UPLOAD PHOTOS] Detailer not found for user ${userId}`);
    return res.status(404).json({
      success: false,
      message: 'Detailer profile not found'
    });
  }

  const detailerId = detailers[0].id;
  console.log(`[UPLOAD PHOTOS] Found detailer ${detailerId}`);

  // Check existing photos count
  const [existingPhotos] = await db.query(
    'SELECT COUNT(*) as count FROM detailer_photos WHERE detailer_id = ? AND photo_type = "gallery" AND is_deleted = 0',
    [detailerId]
  );

  const currentCount = existingPhotos[0].count;
  const newPhotosCount = photos.length;

  console.log(`[UPLOAD PHOTOS] Current photos: ${currentCount}, Uploading: ${newPhotosCount}`);

  if (currentCount + newPhotosCount > 4) {
    console.log(`[UPLOAD PHOTOS] Quota exceeded: ${currentCount} + ${newPhotosCount} > 4`);
    return res.status(400).json({
      success: false,
      message: `Cannot upload ${newPhotosCount} photos. You can only have 4 photos total.`
    });
  }

  // Insert photos into database
  const photoData = photos.map((photo, index) => ({
    detailer_id: detailerId,
    photo_url: `/uploads/detailer-photos/${photo.filename}`,
    photo_type: 'gallery',
    sort_order: currentCount + index + 1
  }));

  try {
    await db.query(
      'INSERT INTO detailer_photos (detailer_id, photo_url, photo_type, sort_order) VALUES ?',
      [photoData.map(p => [p.detailer_id, p.photo_url, p.photo_type, p.sort_order])]
    );
    console.log(`[UPLOAD PHOTOS] Successfully inserted ${newPhotosCount} photos`);
  } catch (error) {
    console.error(`[UPLOAD PHOTOS] Database error:`, error);
    throw error;
  }

  // If there is no primary photo yet, make the first uploaded one primary
  const [primaries] = await db.query(
    'SELECT id FROM detailer_photos WHERE detailer_id = ? AND photo_type = "gallery" AND is_deleted = 0 AND is_primary = 1 LIMIT 1',
    [detailerId]
  );
  if (primaries.length === 0 && photoData.length > 0) {
    const [newRow] = await db.query(
      'SELECT id, photo_url FROM detailer_photos WHERE detailer_id = ? AND photo_url = ? LIMIT 1',
      [detailerId, photoData[0].photo_url]
    );
    if (newRow.length > 0) {
      await db.query('UPDATE detailer_photos SET is_primary = TRUE WHERE id = ?', [newRow[0].id]);
    }
  }

  res.json({
    success: true,
    message: `${newPhotosCount} photo(s) uploaded successfully`,
    data: photoData
  });
});

/**
 * @desc    Get detailer photos
 * @route   GET /api/v1/detailers/:id/photos
 * @access  Public
 */
exports.getPhotos = asyncHandler(async (req, res) => {
  const { id } = req.params;

  const [photos] = await db.query(
    `SELECT id, photo_url, photo_type, is_primary, sort_order, created_at
     FROM detailer_photos 
     WHERE detailer_id = ? AND photo_type = 'gallery' AND is_deleted = 0
     ORDER BY sort_order ASC`,
    [id]
  );

  res.json({
    success: true,
    data: photos
  });
});

/**
 * @desc    Delete detailer photo
 * @route   DELETE /api/v1/detailers/photos/:photoId
 * @access  Private (Detailer only)
 */
exports.deletePhoto = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { photoId } = req.params;

  console.log(`[DELETE PHOTO] User ${userId}, PhotoId ${photoId}`);

  // Get detailer ID
  const [detailers] = await db.query(
    'SELECT id FROM detailers WHERE user_id = ?',
    [userId]
  );

  if (detailers.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Detailer profile not found'
    });
  }

  const detailerId = detailers[0].id;

  // Check if photo exists and belongs to the detailer
  const [photos] = await db.query(
    'SELECT id, photo_url, sort_order FROM detailer_photos WHERE id = ? AND detailer_id = ? AND photo_type = "gallery" AND is_deleted = 0',
    [photoId, detailerId]
  );

  if (photos.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Photo not found'
    });
  }

  const deletedSortOrder = photos[0].sort_order;
  const photoUrl = photos[0].photo_url;

  // Delete the physical file from file system
  const fs = require('fs');
  const path = require('path');
  const filePath = path.join(__dirname, '../../', photoUrl);
  
  if (fs.existsSync(filePath)) {
    try {
      fs.unlinkSync(filePath);
      console.log(`[DELETE PHOTO] Deleted file: ${filePath}`);
    } catch (err) {
      console.error(`[DELETE PHOTO] Error deleting file:`, err);
      // Continue with database deletion even if file deletion fails
    }
  } else {
    console.log(`[DELETE PHOTO] File not found: ${filePath}`);
  }

  // Permanently delete the photo from database
  await db.query(
    'DELETE FROM detailer_photos WHERE id = ?',
    [photoId]
  );

  // Reorder remaining photos
  await db.query(
    `UPDATE detailer_photos 
     SET sort_order = sort_order - 1 
     WHERE detailer_id = ? AND photo_type = "gallery" AND is_deleted = 0 AND sort_order > ?`,
    [detailerId, deletedSortOrder]
  );

  console.log(`[DELETE PHOTO] Success - Permanently deleted photo ${photoId}, reordered remaining photos`);

  res.json({
    success: true,
    message: 'Photo deleted successfully'
  });
});

/**
 * @desc    Set primary photo
 * @route   PUT /api/v1/detailers/photos/:photoId/primary
 * @access  Private (Detailer only)
 */
exports.setPrimaryPhoto = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { photoId } = req.params;

  // Get detailer ID
  const [detailers] = await db.query(
    'SELECT id FROM detailers WHERE user_id = ?',
    [userId]
  );

  if (detailers.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Detailer profile not found'
    });
  }

  const detailerId = detailers[0].id;

  // Check if photo exists and belongs to the detailer
  const [photos] = await db.query(
    'SELECT id FROM detailer_photos WHERE id = ? AND detailer_id = ? AND photo_type = "gallery"',
    [photoId, detailerId]
  );

  if (photos.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Photo not found'
    });
  }

  // Remove primary status from all photos
  await db.query(
    'UPDATE detailer_photos SET is_primary = FALSE WHERE detailer_id = ? AND photo_type = "gallery"',
    [detailerId]
  );

  // Set this photo as primary
  await db.query(
    'UPDATE detailer_photos SET is_primary = TRUE WHERE id = ?',
    [photoId]
  );

  res.json({
    success: true,
    message: 'Primary photo updated successfully'
  });
});

/**
 * @desc    Get detailer payout settings
 * @route   GET /api/v1/detailers/payment-settings
 * @access  Private (Detailer only)
 */
exports.getPaymentSettings = asyncHandler(async (req, res) => {
  const detailerId = await getDetailerIdByUser(req.user.id);
  if (!detailerId) {
    return res.status(404).json({ success: false, message: 'Detailer profile not found' });
  }

  await runAutoPayoutIfDue(detailerId);

  const [rows] = await db.query(
    `SELECT payout_method, mobile_provider, account_name, phone_number, bank_name, account_number,
            auto_payout_enabled, auto_payout_frequency, auto_payout_day_of_week, auto_payout_day_of_month,
            auto_payout_min_amount, updated_at
     FROM detailer_payout_settings
     WHERE detailer_id = ?
     LIMIT 1`,
    [detailerId]
  );

  res.json({
    success: true,
    data:
      rows[0] || {
        payout_method: 'mobile_money',
        mobile_provider: 'airtel_money',
        account_name: '',
        phone_number: '',
        bank_name: '',
        account_number: '',
        auto_payout_enabled: 0,
        auto_payout_frequency: 'weekly',
        auto_payout_day_of_week: 1,
        auto_payout_day_of_month: 1,
        auto_payout_min_amount: 0,
        updated_at: null,
      },
  });
});

/**
 * @desc    Update detailer payout settings
 * @route   PUT /api/v1/detailers/payment-settings
 * @access  Private (Detailer only)
 */
exports.updatePaymentSettings = asyncHandler(async (req, res) => {
  const detailerId = await getDetailerIdByUser(req.user.id);
  if (!detailerId) {
    return res.status(404).json({ success: false, message: 'Detailer profile not found' });
  }

  const payoutMethod = req.body?.payout_method;
  const mobileProvider = req.body?.mobile_provider || null;
  const accountName = String(req.body?.account_name || '').trim();
  const phoneNumber = normalizePhone(req.body?.phone_number || '');
  const bankName = String(req.body?.bank_name || '').trim();
  const accountNumber = String(req.body?.account_number || '').trim();
  const autoEnabled = req.body?.auto_payout_enabled ? 1 : 0;
  const autoFrequency = req.body?.auto_payout_frequency || 'weekly';
  const autoDayOfWeek = Number(req.body?.auto_payout_day_of_week ?? 1);
  const autoDayOfMonth = Number(req.body?.auto_payout_day_of_month ?? 1);
  const autoMinAmount = Number(req.body?.auto_payout_min_amount ?? 0);

  if (!['mobile_money', 'bank'].includes(payoutMethod)) {
    return res.status(400).json({ success: false, message: 'Invalid payout method' });
  }

  if (payoutMethod === 'mobile_money') {
    if (!accountName || !phoneNumber || !mobileProvider) {
      return res.status(400).json({
        success: false,
        message: 'Account name, mobile provider and phone number are required',
      });
    }
  }

  if (payoutMethod === 'bank') {
    if (!accountName || !bankName || !accountNumber) {
      return res.status(400).json({
        success: false,
        message: 'Account name, bank name and account number are required',
      });
    }
  }

  if (!['daily', 'weekly', 'monthly'].includes(autoFrequency)) {
    return res.status(400).json({ success: false, message: 'Invalid automation frequency' });
  }

  await db.query(
    `INSERT INTO detailer_payout_settings (
      detailer_id, payout_method, mobile_provider, account_name, phone_number, bank_name, account_number,
      auto_payout_enabled, auto_payout_frequency, auto_payout_day_of_week, auto_payout_day_of_month, auto_payout_min_amount
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ON DUPLICATE KEY UPDATE
      payout_method = VALUES(payout_method),
      mobile_provider = VALUES(mobile_provider),
      account_name = VALUES(account_name),
      phone_number = VALUES(phone_number),
      bank_name = VALUES(bank_name),
      account_number = VALUES(account_number),
      auto_payout_enabled = VALUES(auto_payout_enabled),
      auto_payout_frequency = VALUES(auto_payout_frequency),
      auto_payout_day_of_week = VALUES(auto_payout_day_of_week),
      auto_payout_day_of_month = VALUES(auto_payout_day_of_month),
      auto_payout_min_amount = VALUES(auto_payout_min_amount),
      updated_at = CURRENT_TIMESTAMP`,
    [
      detailerId,
      payoutMethod,
      payoutMethod === 'mobile_money' ? mobileProvider : null,
      accountName,
      payoutMethod === 'mobile_money' ? phoneNumber : null,
      payoutMethod === 'bank' ? bankName : null,
      payoutMethod === 'bank' ? accountNumber : null,
      autoEnabled,
      autoFrequency,
      autoDayOfWeek,
      autoDayOfMonth,
      Number.isFinite(autoMinAmount) ? Math.max(0, autoMinAmount) : 0,
    ]
  );

  res.json({
    success: true,
    message: 'Payment settings updated successfully',
  });
});

/**
 * @desc    Get payout wallet summary
 * @route   GET /api/v1/detailers/payouts/summary
 * @access  Private (Detailer only)
 */
exports.getPayoutSummary = asyncHandler(async (req, res) => {
  const detailerId = await getDetailerIdByUser(req.user.id);
  if (!detailerId) {
    return res.status(404).json({ success: false, message: 'Detailer profile not found' });
  }

  await runAutoPayoutIfDue(detailerId);
  const balance = await getPayoutBalance(detailerId);

  const [recent] = await db.query(
    `SELECT id, amount, request_type, status, notes, requested_at, processed_at
     FROM payout_requests
     WHERE detailer_id = ?
     ORDER BY requested_at DESC
     LIMIT 10`,
    [detailerId]
  );

  res.json({
    success: true,
    data: {
      available_balance: balance.available,
      total_credited: balance.credited,
      total_requested: balance.requested,
      recent_requests: recent,
    },
  });
});

/**
 * @desc    Request manual payout
 * @route   POST /api/v1/detailers/payouts/request
 * @access  Private (Detailer only)
 */
exports.requestPayout = asyncHandler(async (req, res) => {
  const detailerId = await getDetailerIdByUser(req.user.id);
  if (!detailerId) {
    return res.status(404).json({ success: false, message: 'Detailer profile not found' });
  }

  const amount = Number(req.body?.amount || 0);
  if (!Number.isFinite(amount) || amount <= 0) {
    return res.status(400).json({ success: false, message: 'Enter a valid amount' });
  }

  const [settingsRows] = await db.query(
    `SELECT id
     FROM detailer_payout_settings
     WHERE detailer_id = ?
       AND account_name IS NOT NULL
       AND account_name <> ''
     LIMIT 1`,
    [detailerId]
  );
  if (settingsRows.length === 0) {
    return res.status(400).json({
      success: false,
      message: 'Please complete payment receiving settings before requesting payout',
    });
  }

  const balance = await getPayoutBalance(detailerId);
  if (amount > balance.available) {
    return res.status(400).json({
      success: false,
      message: `Requested amount exceeds available balance (${balance.available.toFixed(2)})`,
    });
  }

  await db.query(
    `INSERT INTO payout_requests (detailer_id, amount, request_type, status, notes, requested_at)
     VALUES (?, ?, 'manual', 'pending', ?, NOW())`,
    [detailerId, amount, 'Manual payout request']
  );

  res.status(201).json({
    success: true,
    message: 'Payout request submitted successfully',
  });
});