const db = require('../config/database');
const { asyncHandler } = require('../middleware/errorHandler');

/**
 * @desc    Get client profile
 * @route   GET /api/v1/clients/profile
 * @access  Private (Client only)
 */
exports.getProfile = asyncHandler(async (req, res) => {
  const userId = req.user.id;

  const [clients] = await db.query(
    `SELECT 
      c.*, u.email, u.created_at as member_since
    FROM clients c
    INNER JOIN users u ON c.user_id = u.id
    WHERE u.id = ? AND c.deleted_at IS NULL`,
    [userId]
  );

  if (clients.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Client profile not found'
    });
  }

  res.json({
    success: true,
    data: clients[0]
  });
});

/**
 * @desc    Update client profile
 * @route   PUT /api/v1/clients/profile
 * @access  Private (Client only)
 */
exports.updateProfile = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const {
    first_name,
    last_name,
    phone,
    address,
    city,
    state,
    latitude,
    longitude
  } = req.body;

  // Prepare update data
  const updateData = {
    first_name,
    last_name,
    phone,
    address,
    city,
    state,
    latitude,
    longitude
  };

  // Add profile image if uploaded
  if (req.file) {
    updateData.profile_image = `/uploads/profile-images/${req.file.filename}`;
  }

  // Build update query
  const fields = Object.keys(updateData).filter(key => updateData[key] !== undefined);
  const values = fields.map(key => updateData[key]);
  values.push(userId);

  const setClause = fields.map(key => `${key} = ?`).join(', ');

  await db.query(
    `UPDATE clients SET ${setClause} WHERE user_id = ?`,
    values
  );

  res.json({
    success: true,
    message: 'Profile updated successfully'
  });
});

/**
 * @desc    Get client dashboard statistics
 * @route   GET /api/v1/clients/dashboard/stats
 * @access  Private (Client only)
 */
exports.getDashboardStats = asyncHandler(async (req, res) => {
  const userId = req.user.id;

  // Get client ID
  const [clients] = await db.query(
    'SELECT id FROM clients WHERE user_id = ?',
    [userId]
  );

  if (clients.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Client profile not found'
    });
  }

  const clientId = clients[0].id;

  // Get booking statistics
  const [bookingStats] = await db.query(
    `SELECT 
      COUNT(*) as total_bookings,
      SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_bookings,
      SUM(CASE WHEN status = 'accepted' THEN 1 ELSE 0 END) as accepted_bookings,
      SUM(CASE WHEN status = 'assigned' THEN 1 ELSE 0 END) as in_progress_bookings,
      SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_bookings,
      SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END) as total_spent
    FROM bookings
    WHERE client_id = ? AND deleted_at IS NULL`,
    [clientId]
  );

  // Get recent bookings
  const [recentBookings] = await db.query(
    `SELECT 
      b.id, b.booking_number, b.status, b.booking_date, b.booking_time, b.amount,
      b.discount_amount,
      d.business_name as detailer_name,
      s.name as service_name,
      o.title as offer_title,
      p.status as payment_status
    FROM bookings b
    INNER JOIN detailers d ON b.detailer_id = d.id
    INNER JOIN services s ON b.service_id = s.id
    LEFT JOIN offers o ON b.offer_id = o.id
    LEFT JOIN payments p ON b.id = p.booking_id
    WHERE b.client_id = ? AND b.deleted_at IS NULL
    ORDER BY b.created_at DESC
    LIMIT 5`,
    [clientId]
  );

  res.json({
    success: true,
    data: {
      ...bookingStats[0],
      recent_bookings: recentBookings
    }
  });
});
