const db = require('../config/database');
const { asyncHandler } = require('../middleware/errorHandler');
const { sendPushToUser } = require('../services/notificationService');

/**
 * Generate unique booking number
 */
const generateBookingNumber = () => {
  const timestamp = Date.now().toString(36).toUpperCase();
  const random = Math.random().toString(36).substring(2, 6).toUpperCase();
  return `BK-${timestamp}-${random}`;
};

/**
 * Calculate haversine distance in KM
 */
const haversineKm = (lat1, lon1, lat2, lon2) => {
  const toRad = (deg) => (deg * Math.PI) / 180;
  const R = 6371;
  const dLat = toRad(lat2 - lat1);
  const dLon = toRad(lon2 - lon1);
  const a =
    Math.sin(dLat / 2) * Math.sin(dLat / 2) +
    Math.cos(toRad(lat1)) * Math.cos(toRad(lat2)) *
    Math.sin(dLon / 2) * Math.sin(dLon / 2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
  return R * c;
};

/**
 * @desc    Create new booking
 * @route   POST /api/v1/bookings
 * @access  Private (Client only)
 */
exports.createBooking = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const {
    service_id,
    offer_id,
    booking_date,
    booking_time,
    vehicle_type,
    vehicle_model,
    vehicle_plate,
    location_address,
    location_latitude,
    location_longitude,
    service_location,
    notes
  } = req.body;

  // Validate required fields
  if (!service_id || !booking_date || !booking_time) {
    return res.status(400).json({
      success: false,
      message: 'Missing required fields: service_id, booking_date, booking_time'
    });
  }

  // Use defaults for optional location fields
  const locAddress = location_address || 'To be confirmed';
  const locLatitude = location_latitude != null ? Number(location_latitude) : null;
  const locLongitude = location_longitude != null ? Number(location_longitude) : null;

  const connection = await db.getConnection();

  try {
    await connection.beginTransaction();

    // Get client ID
    const [clients] = await connection.query(
      'SELECT id FROM clients WHERE user_id = ? AND deleted_at IS NULL',
      [userId]
    );

    if (clients.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Client profile not found'
      });
    }

    const clientId = clients[0].id;

    // Get service + detailer details (for travel distance/fee)
    const [services] = await connection.query(
      `SELECT s.id, s.detailer_id, s.price, d.latitude AS detailer_latitude, d.longitude AS detailer_longitude
       FROM services s
       INNER JOIN detailers d ON s.detailer_id = d.id
       WHERE s.id = ? AND s.is_active = TRUE AND s.deleted_at IS NULL AND d.deleted_at IS NULL`,
      [service_id]
    );

    if (services.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Service not found or not available'
      });
    }

    const service = services[0];

    // Get commission rate from settings
    const [settings] = await connection.query(
      'SELECT key_value FROM settings WHERE key_name = "commission_rate"'
    );

    const commissionRate = settings.length > 0 ? parseFloat(settings[0].key_value) : 10.00;

    // Get travel fee rate from settings (per KM)
    const [travelFeeSettings] = await connection.query(
      'SELECT key_value FROM settings WHERE key_name = "travel_fee_per_km"'
    );
    const travelFeePerKm = travelFeeSettings.length > 0 ? parseFloat(travelFeeSettings[0].key_value) : 0.0;

    // Apply offer discount (if provided)
    let appliedOfferId = null;
    let discountAmount = 0.0;
    let discountedServicePrice = Number(service.price);

    if (offer_id) {
      const [offers] = await connection.query(
        `SELECT o.*
         FROM offers o
         WHERE o.id = ?
           AND o.service_id = ?
           AND o.detailer_id = ?
           AND o.is_active = 1
           AND o.deleted_at IS NULL
           AND (o.starts_at IS NULL OR o.starts_at <= NOW())
           AND (o.ends_at IS NULL OR o.ends_at >= NOW())
         LIMIT 1`,
        [offer_id, service_id, service.detailer_id]
      );

      if (offers.length === 0) {
        await connection.rollback();
        return res.status(400).json({
          success: false,
          message: 'Offer is not available for this service'
        });
      }

      const offer = offers[0];

      if (offer.new_users_only) {
        const [existing] = await connection.query(
          'SELECT COUNT(*) as cnt FROM bookings WHERE client_id = ? AND deleted_at IS NULL',
          [clientId]
        );
        if ((existing[0]?.cnt || 0) > 0) {
          await connection.rollback();
          return res.status(400).json({
            success: false,
            message: 'This offer is only available for new users'
          });
        }
      }

      appliedOfferId = offer.id;

      const original = Number(service.price);
      if (offer.discount_type === 'free') {
        discountAmount = original;
      } else if (offer.discount_type === 'percent') {
        const pct = Math.max(0, Math.min(100, Number(offer.discount_value) || 0));
        discountAmount = (original * pct) / 100;
      } else if (offer.discount_type === 'fixed') {
        discountAmount = Math.max(0, Number(offer.discount_value) || 0);
      }

      // Cap discount so it never makes service negative
      discountAmount = Math.min(original, discountAmount);
      discountAmount = Math.round(discountAmount * 100) / 100;
      discountedServicePrice = Math.max(0, original - discountAmount);
      discountedServicePrice = Math.round(discountedServicePrice * 100) / 100;
    }

    // Calculate travel fee only when detailer must travel (service at client location) and we have coordinates
    const detailerLat = Number(service.detailer_latitude);
    const detailerLng = Number(service.detailer_longitude);
    const hasDetailerCoords = Number.isFinite(detailerLat) && Number.isFinite(detailerLng);
    const hasClientCoords = Number.isFinite(locLatitude) && Number.isFinite(locLongitude);
    const isDetailerTravelling = (service_location || 'client_location') === 'client_location';

    let travelDistanceKm = null;
    let travelFee = 0.0;
    if (isDetailerTravelling && travelFeePerKm > 0 && hasDetailerCoords && hasClientCoords) {
      travelDistanceKm = haversineKm(locLatitude, locLongitude, detailerLat, detailerLng);
      // round to 1 decimal km for display, but compute fee from rounded distance for predictability
      const roundedKm = Math.round(travelDistanceKm * 10) / 10;
      travelDistanceKm = roundedKm;
      travelFee = Math.max(0, roundedKm * travelFeePerKm);
      // round to 2 decimals for currency
      travelFee = Math.round(travelFee * 100) / 100;
    }

    // Commission on discounted service price (travel fee is paid to detailer)
    const commissionAmount = (discountedServicePrice * commissionRate) / 100;
    const totalAmount = Number(discountedServicePrice) + Number(travelFee || 0);

    // Generate booking number
    const bookingNumber = generateBookingNumber();

    // Create booking
    const [result] = await connection.query(
      `INSERT INTO bookings (
        booking_number, client_id, service_id, offer_id, detailer_id, 
        booking_date, booking_time, vehicle_type, vehicle_model, vehicle_plate,
        location_address, location_latitude, location_longitude, service_location, notes,
        amount, commission_rate, commission_amount, discount_amount, travel_distance_km, travel_fee, status
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending')`,
      [
        bookingNumber, clientId, service_id, appliedOfferId, service.detailer_id,
        booking_date, booking_time, vehicle_type || null, vehicle_model || null, vehicle_plate || null,
        locAddress, locLatitude, locLongitude, service_location || 'client_location', notes || null,
        totalAmount, commissionRate, commissionAmount, discountAmount, travelDistanceKm, travelFee
      ]
    );

    const bookingId = result.insertId;

    // Update detailer total bookings
    await connection.query(
      'UPDATE detailers SET total_bookings = total_bookings + 1 WHERE id = ?',
      [service.detailer_id]
    );

    // Create notification for detailer
    const [detailers] = await connection.query(
      'SELECT user_id, business_name FROM detailers WHERE id = ?',
      [service.detailer_id]
    );

    if (detailers.length > 0) {
      await connection.query(
        `INSERT INTO notifications (user_id, type, title, message, data) 
         VALUES (?, 'booking_created', 'New Booking Request', ?, ?)`,
        [
          detailers[0].user_id,
          `You have a new booking request (${bookingNumber})`,
          JSON.stringify({ booking_id: bookingId, booking_number: bookingNumber })
        ]
      );
    }

    await connection.commit();

    if (detailers.length > 0) {
      try {
        await sendPushToUser({
          userId: detailers[0].user_id,
          type: 'booking_created',
          title: 'New Booking Request',
          message: `You have a new booking request (${bookingNumber})`,
          data: { booking_id: bookingId, booking_number: bookingNumber },
        });
      } catch (pushError) {
        console.error('[PUSH] Failed to send booking_created push:', pushError.message);
      }
    }

    res.status(201).json({
      success: true,
      message: 'Booking created successfully. Please upload payment proof to proceed.',
      data: {
        id: bookingId,
        booking_number: bookingNumber
      }
    });

  } catch (error) {
    await connection.rollback();
    throw error;
  } finally {
    connection.release();
  }
});

/**
 * @desc    Accept booking (detailer accepts booking request)
 * @route   PUT /api/v1/bookings/:id/accept
 * @access  Private (Detailer only)
 */
exports.acceptBooking = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const userId = req.user.id;

  const connection = await db.getConnection();

  try {
    await connection.beginTransaction();

    // Verify detailer access
    const [bookings] = await connection.query(
      `SELECT b.*, c.user_id as client_user_id FROM bookings b
       INNER JOIN detailers d ON b.detailer_id = d.id
       INNER JOIN clients c ON b.client_id = c.id
       WHERE b.id = ? AND d.user_id = ? AND b.deleted_at IS NULL`,
      [id, userId]
    );

    if (bookings.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Booking not found or access denied'
      });
    }

    const booking = bookings[0];

    if (booking.status !== 'pending') {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Only pending bookings can be accepted'
      });
    }

    // Update booking status to accepted
    await connection.query(
      'UPDATE bookings SET status = "accepted", updated_at = NOW() WHERE id = ?',
      [id]
    );

    // Notify client
    await connection.query(
      `INSERT INTO notifications (user_id, type, title, message, data) 
       VALUES (?, 'booking_accepted', 'Booking Accepted', ?, ?)`,
      [
        booking.client_user_id,
        `Your booking ${booking.booking_number} has been accepted by the detailer.`,
        JSON.stringify({ booking_id: id, booking_number: booking.booking_number })
      ]
    );

    await connection.commit();

    try {
      await sendPushToUser({
        userId: booking.client_user_id,
        type: 'booking_accepted',
        title: 'Booking Accepted',
        message: `Your booking ${booking.booking_number} has been accepted by the detailer.`,
        data: { booking_id: Number(id), booking_number: booking.booking_number },
      });
    } catch (pushError) {
      console.error('[PUSH] Failed to send booking_accepted push:', pushError.message);
    }

    res.json({
      success: true,
      message: 'Booking accepted successfully'
    });

  } catch (error) {
    await connection.rollback();
    throw error;
  } finally {
    connection.release();
  }
});

/**
 * @desc    Reject booking (detailer rejects booking request)
 * @route   PUT /api/v1/bookings/:id/reject
 * @access  Private (Detailer only)
 */
exports.rejectBooking = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const { reason } = req.body;
  const userId = req.user.id;

  const connection = await db.getConnection();

  try {
    await connection.beginTransaction();

    // Verify detailer access
    const [bookings] = await connection.query(
      `SELECT b.*, c.user_id as client_user_id FROM bookings b
       INNER JOIN detailers d ON b.detailer_id = d.id
       INNER JOIN clients c ON b.client_id = c.id
       WHERE b.id = ? AND d.user_id = ? AND b.deleted_at IS NULL`,
      [id, userId]
    );

    if (bookings.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Booking not found or access denied'
      });
    }

    const booking = bookings[0];

    if (booking.status !== 'pending') {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Only pending bookings can be rejected'
      });
    }

    // Update booking status to cancelled (schema enum does not include "rejected")
    await connection.query(
      'UPDATE bookings SET status = "cancelled", cancelled_at = NOW(), cancellation_reason = ?, updated_at = NOW() WHERE id = ?',
      [reason || 'Rejected by detailer', id]
    );

    // Notify client
    await connection.query(
      `INSERT INTO notifications (user_id, type, title, message, data) 
       VALUES (?, 'booking_rejected', 'Booking Rejected', ?, ?)`,
      [
        booking.client_user_id,
        `Your booking ${booking.booking_number} has been rejected. ${reason || ''}`,
        JSON.stringify({ booking_id: id, booking_number: booking.booking_number, reason })
      ]
    );

    await connection.commit();

    try {
      await sendPushToUser({
        userId: booking.client_user_id,
        type: 'booking_rejected',
        title: 'Booking Rejected',
        message: `Your booking ${booking.booking_number} has been rejected. ${reason || ''}`.trim(),
        data: { booking_id: Number(id), booking_number: booking.booking_number, reason },
      });
    } catch (pushError) {
      console.error('[PUSH] Failed to send booking_rejected push:', pushError.message);
    }

    res.json({
      success: true,
      message: 'Booking rejected'
    });

  } catch (error) {
    await connection.rollback();
    throw error;
  } finally {
    connection.release();
  }
});

/**
 * @desc    Start booking (detailer marks booking as in progress)
 * @route   PUT /api/v1/bookings/:id/start
 * @access  Private (Detailer only)
 */
exports.startBooking = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const userId = req.user.id;

  const connection = await db.getConnection();

  try {
    await connection.beginTransaction();

    // Verify detailer access
    const [bookings] = await connection.query(
      `SELECT b.*, c.user_id as client_user_id FROM bookings b
       INNER JOIN detailers d ON b.detailer_id = d.id
       INNER JOIN clients c ON b.client_id = c.id
       WHERE b.id = ? AND d.user_id = ? AND b.deleted_at IS NULL`,
      [id, userId]
    );

    if (bookings.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Booking not found or access denied'
      });
    }

    const booking = bookings[0];

    if (booking.status !== 'accepted') {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Only accepted bookings can be started'
      });
    }

    // Update booking status to assigned (in progress)
    await connection.query(
      'UPDATE bookings SET status = "assigned", updated_at = NOW() WHERE id = ?',
      [id]
    );

    // Notify client
    await connection.query(
      `INSERT INTO notifications (user_id, type, title, message, data) 
       VALUES (?, 'booking_started', 'Service Started', ?, ?)`,
      [
        booking.client_user_id,
        `Your car wash service has started (${booking.booking_number}).`,
        JSON.stringify({ booking_id: id, booking_number: booking.booking_number })
      ]
    );

    await connection.commit();

    try {
      await sendPushToUser({
        userId: booking.client_user_id,
        type: 'booking_started',
        title: 'Service Started',
        message: `Your car wash service has started (${booking.booking_number}).`,
        data: { booking_id: Number(id), booking_number: booking.booking_number },
      });
    } catch (pushError) {
      console.error('[PUSH] Failed to send booking_started push:', pushError.message);
    }

    res.json({
      success: true,
      message: 'Booking started'
    });

  } catch (error) {
    await connection.rollback();
    throw error;
  } finally {
    connection.release();
  }
});

/**
 * @desc    Get client bookings
 * @route   GET /api/v1/bookings/my-bookings
 * @access  Private (Client only)
 */
exports.getClientBookings = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { status, page = 1, limit = 20 } = req.query;

  // Get client ID
  const [clients] = await db.query(
    'SELECT id FROM clients WHERE user_id = ?',
    [userId]
  );

  if (clients.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Client profile not found'
    });
  }

  const clientId = clients[0].id;

  let query = `
    SELECT 
      b.*, s.name AS service_name, s.duration_minutes,
      d.business_name, d.phone AS detailer_phone, d.address AS detailer_address,
      p.status AS payment_status, p.payment_method
    FROM bookings b
    INNER JOIN services s ON b.service_id = s.id
    INNER JOIN detailers d ON b.detailer_id = d.id
    LEFT JOIN payments p ON b.id = p.booking_id
    WHERE b.client_id = ? AND b.deleted_at IS NULL
  `;

  const params = [clientId];

  if (status) {
    query += ' AND b.status = ?';
    params.push(status);
  }

  query += ' ORDER BY b.created_at DESC LIMIT ? OFFSET ?';
  const offset = (parseInt(page) - 1) * parseInt(limit);
  params.push(parseInt(limit), offset);

  const [bookings] = await db.query(query, params);

  res.json({
    success: true,
    data: bookings
  });
});

/**
 * @desc    Get assigned bookings for detailer
 * @route   GET /api/v1/bookings/assigned
 * @access  Private (Detailer only)
 */
exports.getAssignedBookings = asyncHandler(async (req, res) => {
  const userId = req.user.id;
  const { status, page = 1, limit = 20 } = req.query;

  // Get detailer ID
  const [detailers] = await db.query(
    'SELECT id FROM detailers WHERE user_id = ?',
    [userId]
  );

  if (detailers.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Detailer profile not found'
    });
  }

  const detailerId = detailers[0].id;

  let query = `
    SELECT 
      b.*, s.name AS service_name, s.duration_minutes,
      CONCAT(c.first_name, ' ', c.last_name) AS client_name, c.phone AS client_phone, u.email AS client_email, c.address AS client_address,
      d.business_name AS detailer_business_name, d.owner_name AS detailer_owner_name, d.phone AS detailer_phone, d.address AS detailer_address, du.email AS detailer_email,
      p.status AS payment_status
    FROM bookings b
    INNER JOIN services s ON b.service_id = s.id
    INNER JOIN clients c ON b.client_id = c.id
    INNER JOIN users u ON c.user_id = u.id
    INNER JOIN detailers d ON b.detailer_id = d.id
    INNER JOIN users du ON d.user_id = du.id
    LEFT JOIN payments p ON b.id = p.booking_id
    WHERE b.detailer_id = ? AND b.deleted_at IS NULL
  `;

  const params = [detailerId];

  if (status) {
    query += ' AND b.status = ?';
    params.push(status);
  }

  query += ' ORDER BY b.booking_date ASC, b.booking_time ASC LIMIT ? OFFSET ?';
  const offset = (parseInt(page) - 1) * parseInt(limit);
  params.push(parseInt(limit), offset);

  const [bookings] = await db.query(query, params);

  res.json({
    success: true,
    data: bookings
  });
});

/**
 * @desc    Get booking by ID
 * @route   GET /api/v1/bookings/:id
 * @access  Private
 */
exports.getBookingById = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const userId = req.user.id;
  const userRole = req.user.role;

  let query = `
    SELECT 
      b.*, s.name AS service_name, s.duration_minutes, s.price,
      d.business_name, d.phone AS detailer_phone, d.address AS detailer_address,
      c.first_name, c.last_name, c.phone AS client_phone, c.address AS client_address, u.email AS client_email,
      o.title AS offer_title, o.discount_type AS offer_discount_type, o.discount_value AS offer_discount_value,
      p.status AS payment_status, p.payment_method, p.payment_proof
    FROM bookings b
    INNER JOIN services s ON b.service_id = s.id
    INNER JOIN detailers d ON b.detailer_id = d.id
    INNER JOIN clients c ON b.client_id = c.id
    INNER JOIN users u ON c.user_id = u.id
    LEFT JOIN offers o ON b.offer_id = o.id
    LEFT JOIN payments p ON b.id = p.booking_id
    WHERE b.id = ? AND b.deleted_at IS NULL
  `;

  const [bookings] = await db.query(query, [id]);

  if (bookings.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Booking not found'
    });
  }

  const booking = bookings[0];

  // Verify access
  if (userRole === 'client') {
    const [clients] = await db.query('SELECT id FROM clients WHERE user_id = ?', [userId]);
    if (clients.length === 0 || clients[0].id !== booking.client_id) {
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }
  } else if (userRole === 'detailer') {
    const [detailers] = await db.query('SELECT id FROM detailers WHERE user_id = ?', [userId]);
    if (detailers.length === 0 || detailers[0].id !== booking.detailer_id) {
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }
  }

  res.json({
    success: true,
    data: booking
  });
});

/**
 * @desc    Complete booking
 * @route   PUT /api/v1/bookings/:id/complete
 * @access  Private (Detailer only)
 */
exports.completeBooking = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const userId = req.user.id;

  const connection = await db.getConnection();

  try {
    await connection.beginTransaction();

    // Verify ownership and status
    const [bookings] = await connection.query(
      `SELECT b.id, b.booking_number, b.client_id, b.detailer_id, b.status, d.user_id 
       FROM bookings b
       INNER JOIN detailers d ON b.detailer_id = d.id
       WHERE b.id = ? AND d.user_id = ? AND b.deleted_at IS NULL`,
      [id, userId]
    );

    if (bookings.length === 0) {
      await connection.rollback();
      return res.status(404).json({
        success: false,
        message: 'Booking not found or access denied'
      });
    }

    const booking = bookings[0];

    if (booking.status !== 'assigned' && booking.status !== 'accepted') {
      await connection.rollback();
      return res.status(400).json({
        success: false,
        message: 'Only assigned or accepted bookings can be marked as completed'
      });
    }

    // Update booking status
    await connection.query(
      'UPDATE bookings SET status = "completed", completed_at = NOW() WHERE id = ?',
      [id]
    );

    // Update detailer completed bookings
    await connection.query(
      'UPDATE detailers SET completed_bookings = completed_bookings + 1 WHERE id = ?',
      [booking.detailer_id]
    );

    // Notify client
    const [clients] = await connection.query(
      'SELECT user_id FROM clients WHERE id = ?',
      [booking.client_id]
    );

    if (clients.length > 0) {
      await connection.query(
        `INSERT INTO notifications (user_id, type, title, message, data) 
         VALUES (?, 'booking_completed', 'Service Completed - Payment Required', ?, ?)`,
        [
          clients[0].user_id,
          `Your car wash service has been completed. Please make payment for booking ${booking.booking_number}.`,
          JSON.stringify({ booking_id: id, booking_number: booking.booking_number, requires_payment: true })
        ]
      );
    }

    await connection.commit();

    if (clients.length > 0) {
      try {
        await sendPushToUser({
          userId: clients[0].user_id,
          type: 'booking_completed',
          title: 'Service Completed - Payment Required',
          message: `Your car wash service has been completed. Please make payment for booking ${booking.booking_number}.`,
          data: { booking_id: Number(id), booking_number: booking.booking_number, requires_payment: true },
        });
      } catch (pushError) {
        console.error('[PUSH] Failed to send booking_completed push:', pushError.message);
      }
    }

    res.json({
      success: true,
      message: 'Booking marked as completed'
    });

  } catch (error) {
    await connection.rollback();
    throw error;
  } finally {
    connection.release();
  }
});

/**
 * @desc    Cancel booking
 * @route   PUT /api/v1/bookings/:id/cancel
 * @access  Private (Client or Detailer)
 */
exports.cancelBooking = asyncHandler(async (req, res) => {
  const { id } = req.params;
  const { cancellation_reason } = req.body;
  const userId = req.user.id;
  const userRole = req.user.role;

  // Verify access
  let accessQuery = '';
  if (userRole === 'client') {
    accessQuery = `
      SELECT b.id, b.status FROM bookings b
      INNER JOIN clients c ON b.client_id = c.id
      WHERE b.id = ? AND c.user_id = ? AND b.deleted_at IS NULL
    `;
  } else if (userRole === 'detailer') {
    accessQuery = `
      SELECT b.id, b.status FROM bookings b
      INNER JOIN detailers d ON b.detailer_id = d.id
      WHERE b.id = ? AND d.user_id = ? AND b.deleted_at IS NULL
    `;
  }

  const [bookings] = await db.query(accessQuery, [id, userId]);

  if (bookings.length === 0) {
    return res.status(404).json({
      success: false,
      message: 'Booking not found or access denied'
    });
  }

  const booking = bookings[0];

  if (booking.status === 'completed' || booking.status === 'cancelled') {
    return res.status(400).json({
      success: false,
      message: `Booking is already ${booking.status}`
    });
  }

  // Cancel booking
  await db.query(
    'UPDATE bookings SET status = "cancelled", cancelled_at = NOW(), cancellation_reason = ? WHERE id = ?',
    [cancellation_reason, id]
  );

  res.json({
    success: true,
    message: 'Booking cancelled successfully'
  });
});
